/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { createImageLoader } from './image_loader';
/**
 * Function that generates an ImageLoader for [Cloudflare Image
 * Resizing](https://developers.cloudflare.com/images/image-resizing/) and turns it into an Angular
 * provider. Note: Cloudflare has multiple image products - this provider is specifically for
 * Cloudflare Image Resizing; it will not work with Cloudflare Images or Cloudflare Polish.
 *
 * @param path Your domain name, e.g. https://mysite.com
 * @param options An object with extra configuration:
 * - `ensurePreconnect`: boolean flag indicating whether the NgOptimizedImage directive
 *                       should verify that there is a corresponding `<link rel="preconnect">`
 *                       present in the document's `<head>`.
 * @returns Provider that provides an ImageLoader function
 *
 * @publicApi
 * @developerPreview
 */
export const provideCloudflareLoader = createImageLoader(createCloudflareUrl, ngDevMode ? ['https://<ZONE>/cdn-cgi/image/<OPTIONS>/<SOURCE-IMAGE>'] : undefined);
function createCloudflareUrl(path, config) {
    let params = `format=auto`;
    if (config.width) {
        params += `,width=${config.width}`;
    }
    // Cloudflare image URLs format:
    // https://developers.cloudflare.com/images/image-resizing/url-format/
    return `${path}/cdn-cgi/image/${params}/${config.src}`;
}
//# sourceMappingURL=data:application/json;base64,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